<?php
/*--------------------------------------------------------------------------------------------------
    ImplementationFactory.php 2020-02-17
    Gambio GmbH
    http://www.gambio.de
    Copyright (c) 2020 Gambio GmbH
    Released under the GNU General Public License (Version 2)
    [http://www.gnu.org/licenses/gpl-2.0.html]
    --------------------------------------------------------------------------------------------------
 */
declare(strict_types=1);

namespace Gambio\Shop\Properties\ProductModifiers\Database;

use CI_DB_query_builder;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Groups\GroupDTOBuilder;
use Gambio\Shop\ProductModifiers\Database\Core\DTO\Modifiers\ModifierDTOBuilder;
use Gambio\Shop\ProductModifiers\Database\Core\Factories\Interfaces\ImplementationFactoryInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Mappers\Interfaces\GroupMapperInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\Interfaces\GroupReaderCompositeInterface;
use Gambio\Shop\ProductModifiers\Database\Core\Readers\Interfaces\ModifierReaderCompositeInterface;
use Gambio\Shop\Properties\ProductModifiers\Database\Builders\PropertyGroupBuilder;
use Gambio\Shop\Properties\ProductModifiers\Database\Builders\PropertyModifierBuilder;
use Gambio\Shop\Properties\ProductModifiers\Database\Readers\PropertyGroupReader;
use Gambio\Shop\Properties\ProductModifiers\Database\Readers\PropertyModifierReader;
use Gambio\Shop\ProductModifiers\Database\Presentation\Mappers\Interfaces\PresentationMapperInterface;
use Gambio\Shop\ProductModifiers\Groups\Builders\GroupBuilderInterface;
use Gambio\Shop\ProductModifiers\Modifiers\Builders\ModifierBuilderInterface;
use StaticGXCoreLoader;

/**
 * Class ImplementationFactory
 * @package Gambio\Shop\Properties\ProductModifiers\Database
 *
 */
class ImplementationFactory implements ImplementationFactoryInterface
{
    /**
     * @var CI_DB_query_builder
     */
    protected $queryBuilder;
    /**
     * @var PresentationMapperInterface
     */
    private $presentationMapper;
    
    
    /**
     * AttributeImplementationFactory constructor.
     *
     * @param PresentationMapperInterface $presentationMapper
     * @param CI_DB_query_builder|null    $queryBuilder
     */
    public function __construct(
        PresentationMapperInterface $presentationMapper,
        CI_DB_query_builder $queryBuilder = null
    ) {
        $this->queryBuilder       = $queryBuilder ? : StaticGXCoreLoader::getDatabaseQueryBuilder();
        $this->presentationMapper = $presentationMapper;
    }
    
    
    /**
     * @inheritDoc
     */
    public function source(): string
    {
        return 'property';
    }
    
    
    /**
     * @inheritDoc
     */
    public function createModifierReader(): ModifierReaderCompositeInterface
    {
        return new PropertyModifierReader($this->queryBuilder, new ModifierDTOBuilder());
    }
    
    
    /**
     * @inheritDoc
     */
    public function createGroupReader(): GroupReaderCompositeInterface
    {
        return new PropertyGroupReader($this->queryBuilder, new GroupDTOBuilder());
    }
    
    
    /**
     * @inheritDoc
     */
    public function createGroupMapper(): GroupMapperInterface
    {
        return new PropertyGroupMapper($this->createGroupBuilder(),
                                       $this->createModifierBuilder(),
                                       $this->presentationMapper);
    }
    
    
    /**
     * @inheritDoc
     */
    protected function createGroupBuilder(): GroupBuilderInterface
    {
        return new PropertyGroupBuilder();
    }
    
    
    /**
     * @inheritDoc
     */
    protected function createModifierBuilder(): ModifierBuilderInterface
    {
        return new PropertyModifierBuilder();
    }
}